--  Copyright (2008-2013) Cdric Coussinet (cedric.coussinet@nomoseed.net)
--
--  This program is free software: you can redistribute it and/or modify
--  it under the terms of the GNU Affero General Public License as published
--  by the Free Software Foundation, either version 3 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
--  GNU Affero General Public License for more details.

--  You should have received a copy of the GNU Affero General Public License
--  along with this program. If not, see <http://www.gnu.org/licenses/>

with Nomo.Interpreter.Plant.Operators;

with Nomo.Interpreter.Plant.Errors_Manager;

with Nomo.Interpreter.Plant.Rule_Buffer.Writing;

with Nomo.Interpreter.Plant.Internal_Pointers_Register;

with Nomo.Interpreter.Types_Directory.Relations;

with Nomo.Interpreter.Event_Memory.Internal.Plant;

with Nomo.Internal_Messages.Logging;

with Nomo.Internal_Messages.Plant;

with Nomo.Interpreter.Maximums_Register;

with Nomo.Premise_Properties;

with Nomo.Reader.Parameters;
pragma Elaborate_All (Nomo.Reader.Parameters);

package body Nomo.Interpreter.Plant.Internal_Chunks is

   Step : constant Positive_Time := Reader.Parameters.Get_Period;

   Last_Operator_Without_Parameter : constant Positive :=  Operators.Operator_Without_Parameter'Pos(Operators.Operator_Without_Parameter'Last);

   procedure Assume (This    : in out Internal_Chunk;
                     T       : in Positive_Time;
                     Pointer : in Internal_Chunk_Index) is
      use Errors_Manager;
      use Operators.Operators_Description;
      use Types_Index.Types_Index_Instance;
      use Internal_Messages.Plant;
      use Types_Directory.Relations;

      procedure Assume_Pointer;
      procedure Assume_Pointer is
      begin
         if Internal_Pointers_Register.Is_Expired (Pointer, T) then
            Error (EXPIRED_INTERNAL_CHUNCK);
         end if;
         Internal_Pointers_Register.Get_Target (Pointer, Targets.Target (This));
         This.Event_Position := Internal_Pointers_Register.Get_Event_Position (Pointer);
      end Assume_Pointer;

      procedure Get_Event (Position : in Internal_Event_Index);
      procedure Get_Event (Position : in Internal_Event_Index) is
      begin
         if  This.Event_Position /= Position then
            Event_Memory.Internal.Plant.Get_Event (This.Index, Position, This.Arrival_Time, This.Data);
         else
            Error (FAIL_TARGETED_AFFECTATION);
         end if;
      end Get_Event;

   begin
      if This.Operation <= Last_Operator_Without_Parameter then
         case Operator_Without_Parameter'Val(This.Operation) is
            when Get_There =>
               Assume_Pointer;
               Event_Memory.Internal.Plant.Get_Event (This.Index, This.Event_Position, This.Data);
            when Get_Before =>
               Assume_Pointer;
               Get_Event (Event_Memory.Internal.Plant.Get_Before_Position (This.Index, This.Event_Position));
            when Get_First =>
               Assume_Pointer;
               Event_Memory.Internal.Plant.Get_Event (This.Index, Event_Memory.Internal.Plant.Get_First_Position (This.Index, This.Event_Position), This.Arrival_Time, This.Data);
            when Get_Last =>
               Assume_Pointer;
               Event_Memory.Internal.Plant.Get_Event (This.Index, Event_Memory.Internal.Plant.Get_Last_Position (This.Index, This.Event_Position), This.Arrival_Time, This.Data);
            when Get_After =>
               Assume_Pointer;
               Get_Event (Event_Memory.Internal.Plant.Get_After_Position (This.Index, This.Event_Position));
            when Get_Same =>
               Assume_Pointer;
               Get_Event (Event_Memory.Internal.Plant.Get_Same_Position (This.Index, This.Event_Position));
            when Create_Conception =>
                  Internal_Pointers_Register.Get_Target (Pointer, Targets.Target (This));
                  This.Arrival_Time := This.Capture_Time;
               if This.Index in Conception_Type_Index then
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
               else
                  Error (ILLEGAL_REFERENCE_FOR_CREATION);
               end if;
            when Create_Reward =>
               Internal_Pointers_Register.Get_Target (Pointer, Targets.Target (This));
               This.Arrival_Time := This.Capture_Time;
               if This.Index in Reward_Type_Index then
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
               else
                  Error (ILLEGAL_REFERENCE_FOR_CREATION);
               end if;
            when Create_Landmark =>
               Internal_Pointers_Register.Get_Target (Pointer, Targets.Target (This));
               This.Arrival_Time := This.Capture_Time;
               if This.Index in Landmark_Type_Index then
                  if Has_Prediction_Linked (This.Index) then
                     Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
                  else
                     Error (ILLEGAL_LANDMARK_TYPE);
                  end if;
               else
                  Error (ILLEGAL_REFERENCE_FOR_CREATION);
               end if;
            when Forward =>
               if This.Is_Expired (T) then
                  Error (AFFECTATION_EXPIRED_FOR_SHIFTING_TIME);
               end if;
               This.Arrival_Time := This.Arrival_Time + Step;
               This.Capture_Time := This.Capture_Time + Step;
            when Backward =>
               if This.Is_Expired (T) then
                  Error (AFFECTATION_EXPIRED_FOR_SHIFTING_TIME);
               end if;
               This.Arrival_Time := This.Arrival_Time - Step;
               This.Capture_Time := This.Capture_Time - Step;
            when Transform_In_Temporal_Reference | Transform_In_Temporal_Reference_And_Origin =>
               if This.Is_Expired (T) then
                  Error (AFFECTATION_EXPIRED_FOR_REFERENCE_TIME);
               end if;
            when Operators.Operators_Description.Target =>
               Error (AFFECTATION_EXPIRED_OPERATOR_FOR_TARGET);
            when others =>
               Error (MISSING_TARGET_FOR_AFFECTATION_OPERATOR);
         end case;
      else
         Error (ILLEGAL_OPERATOR_IN_AFFECTATION);
      end if;
   end Assume;

   procedure Assume_With_Target (This         : in out Internal_Chunk;
                                 Pointer      : in Internal_Chunk_Index;
                                 Target_Chunk : in Internal_Chunk) is
      use Errors_Manager;
      use Operators.Operators_Description;
      use Types_Index.Types_Index_Instance;
      use Internal_Messages.Plant;
      use Types_Directory.Relations;

   begin
      pragma Assert (Target_Chunk.Index /= 0);
      if This.Operation <= Last_Operator_Without_Parameter then
         case Operator_Without_Parameter'Val(This.Operation) is
            when Operators.Affectation_Operators =>
               This.Index := Internal_Pointers_Register.Get_Type_Index (Pointer);
               This.Arrival_Time := Target_Chunk.Get_Arrival_Time (This.Operation);
               This.Capture_Time := Target_Chunk.Capture_Time;
               Event_Memory.Internal.Plant.Get_Event (This.Index, This.Arrival_Time, This.Data);
               if This.Arrival_Time = 0 then
                  Error (FAIL_TARGETED_AFFECTATION);
               end if;
            when Copy =>
               This.Index := Target_Chunk.Index;
               This.Arrival_Time := Target_Chunk.Arrival_Time;
               This.Capture_Time := Target_Chunk.Capture_Time;
               Copy (This.Data, Target_Chunk.Data);
               This.Event_Position := Target_Chunk.Event_Position;
            when Create_Conception =>
               if Target_Chunk.Index in Conception_Type_Index then
                  This.Arrival_Time := Target_Chunk.Arrival_Time;
                  This.Capture_Time := Target_Chunk.Capture_Time;
                  This.Index := Target_Chunk.Index;
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
            when Create_Twin =>
               if Has_Twin_Type (Target_Chunk.Index) then
                  This.Arrival_Time := Target_Chunk.Arrival_Time;
                  This.Capture_Time := Target_Chunk.Capture_Time;
                  Internal_Messages.Logging.Set (This.Data, Target_Chunk.Data);
                  This.Index := Get_Twin_Type (Target_Chunk.Index);
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
            when Create_Check_Forward =>
               if (Target_Chunk.Index in Perception_Type_Index or Target_Chunk.Index in Conception_Type_Index) and then Has_Prediction_Type (Target_Chunk.Index) then
                  This.Arrival_Time := Target_Chunk.Arrival_Time + Step;
                  This.Capture_Time := Target_Chunk.Capture_Time + Step;
                  This.Index := Get_Check_Type (Get_Prediction_Type (Target_Chunk.Index));
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
            when Create_Prediction =>
               if (Target_Chunk.Index in Perception_Type_Index or Target_Chunk.Index in Conception_Type_Index) and then Has_Prediction_Type (Target_Chunk.Index) then
                  This.Arrival_Time := Target_Chunk.Arrival_Time;
                  This.Capture_Time := Target_Chunk.Capture_Time;
                  This.Index := Get_Prediction_Type (Target_Chunk.Index);
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
            when Create_Landmark =>
               if (Target_Chunk.Index in Perception_Type_Index or Target_Chunk.Index in Conception_Type_Index) and then Has_Prediction_Type (Target_Chunk.Index) then
                  This.Arrival_Time := Target_Chunk.Arrival_Time;
                  This.Capture_Time := Target_Chunk.Capture_Time;
                  This.Index := Get_Landmark_Type (Get_Prediction_Type (Target_Chunk.Index));
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
            when Create_Reward =>
               if Has_Reward_Type (Target_Chunk.Index) then
                  This.Arrival_Time := Target_Chunk.Arrival_Time;
                  This.Capture_Time := Target_Chunk.Capture_Time;
                  This.Index := Get_Reward_Type (Target_Chunk.Index);
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
               Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
            when Create_Command =>
               if Target_Chunk.Index in Perception_Type_Index and then Has_Command_Type(Target_Chunk.Index) then
                  This.Arrival_Time := Target_Chunk.Arrival_Time;
                  This.Capture_Time := Target_Chunk.Capture_Time;
                  Set_Information (This.Data, Maximums_Register.Incrementation.Get_Last_Information (Target_Chunk.Index));
                  This.Index := Get_Command_Type (Target_Chunk.Index);
               else
                  Error (ILLEGAL_CREATION_OPERATOR_IN_INTERNAL_CHUNK);
               end if;
            when Same_Time =>
               This.Arrival_Time := Target_Chunk.Arrival_Time;
               This.Capture_Time := Target_Chunk.Capture_Time;
            when Forward =>
               This.Arrival_Time := Target_Chunk.Arrival_Time + Step;
               This.Capture_Time := Target_Chunk.Capture_Time + Step;
            when Backward =>
               This.Arrival_Time := Target_Chunk.Arrival_Time - Step;
               This.Capture_Time := Target_Chunk.Capture_Time - Step;
            when others =>
               raise Constraint_Error;
         end case;
      else
         Error (ILLEGAL_OPERATOR_IN_AFFECTATION);
      end if;
   end Assume_With_Target;

   procedure Create_Perception_Forward (This         : in out Internal_Chunk;
                                        Capture_Time : in Positive_Time;
                                        Arrival_Time : in Positive_Time;
                                        Input_Index  : in Input_Type_Index) is
      use Internal_Messages.Plant;
      use Types_Directory.Relations;
   begin
      This.Capture_Time := Capture_Time + Step;
      This.Arrival_Time := Arrival_Time + Step;
      This.Index := Get_Perception_Type (Input_Index);
      Set_Information (This.Data, Maximums_Register.Incrementation.Get_New_Information (This.Index));
   end Create_Perception_Forward;

   function Get_Arrival_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return This.Arrival_Time;
   end Get_Arrival_Time;

   function Get_Arrival_Time (This      : in Internal_Chunk;
                              Operation : in Positive) return Positive_Time is
      use Errors_Manager;
      use Operators.Operators_Description;

      function Valid (Arrival_Time : in Positive_Time) return Positive_Time;
      function Valid (Arrival_Time : in Positive_Time) return Positive_Time is
      begin
         if Arrival_Time = Positive_Time'Last then
            Error (FAIL_SEARCH_EVENT);
         end if;
         return Arrival_Time;
      end Valid;

   begin
      case Operator_Without_Parameter'Val(Operation) is
         when Get_There =>
            return This.Get_Arrival_Time;
         when Get_Before =>
            return Valid (This.Get_Before_Arrival_Time);
         when Get_First =>
            return This.Get_First_Arrival_Time;
         when Get_Last =>
            return This.Get_Last_Arrival_Time;
         when Get_After =>
            return Valid (This.Get_After_Arrival_Time);
         when Get_Same =>
            return Valid (This.Get_Same_Arrival_Time);
         when others =>
            raise Constraint_Error;
      end case;
   end Get_Arrival_Time;

   function Get_Capture_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return This.Capture_Time;
   end Get_Capture_Time;

   function Get_Before_Arrival_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return Event_Memory.Internal.Plant.Get_Before_Arrival_Time (This.Index, This.Event_Position);
   end Get_Before_Arrival_Time;

   function Get_First_Arrival_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return Event_Memory.Internal.Plant.Get_First_Arrival_Time (This.Index, This.Event_Position);
   end Get_First_Arrival_Time;

   function Get_Last_Arrival_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return Event_Memory.Internal.Plant.Get_Last_Arrival_Time (This.Index, This.Event_Position);
   end Get_Last_Arrival_Time;

   function Get_After_Arrival_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return Event_Memory.Internal.Plant.Get_After_Arrival_Time (This.Index, This.Event_Position);
   end Get_After_Arrival_Time;

   function Get_Same_Arrival_Time (This : in Internal_Chunk) return Positive_Time is
   begin
      return Event_Memory.Internal.Plant.Get_Same_Arrival_Time (This.Index, This.Event_Position);
   end Get_Same_Arrival_Time;

   function Get_Operation (This : in Internal_Chunk) return Positive is
   begin
      return This.Operation;
   end Get_Operation;

   function Is_Expired (This : in Internal_Chunk;
                        T    : in Positive_Time) return Boolean is
   begin
      return Target (This).Is_Expired (T);
   end Is_Expired;

   function Is_Valid (This : in Internal_Chunk) return Boolean;
   pragma Inline (Is_Valid);
   function Is_Valid (This : in Internal_Chunk) return Boolean is
   begin
      return (This.Capture_Time /= 0 or else
        (Operators.Internal_Operators_With_Parameters (This.Operation).Time_Span_Imposed
         and Operators.Internal_Operators_With_Parameters (This.Operation).Information_Imposed
         and Operators.Internal_Operators_With_Parameters (This.Operation).Credibility_Imposed
           and Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed));
   end Is_Valid;

   procedure Send_In_Conclusion (This         : in Internal_Chunk;
                                 Is_Reference : in Boolean) is
      use Types_Index.Types_Index_Instance;
      use Errors_Manager;
      use Operators;
   begin
      if Is_Valid (This) then
         if (not (Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed)
             or else not(Operators.Internal_Operators_With_Parameters (This.Operation).Information_Imposed
                         and This.Index = Operators.Internal_Operators_With_Parameters (This.Operation).Type_Value))
           and This.Operation > Last_Operator_Without_Parameter
           and Operators.Internal_Operators_With_Parameters (This.Operation).Is_Conclusion then
            if Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed then
               Rule_Buffer.Writing.Set_Internal_Conclusion (Operators.Internal_Operators_With_Parameters (This.Operation).Type_Value,
                                                            This.Data,
                                                            This.Arrival_Time,
                                                            This.Operation,
                                                            Is_Reference);
            else
               Rule_Buffer.Writing.Set_Internal_Conclusion (This.Index,
                                                            This.Data,
                                                            This.Arrival_Time,
                                                            This.Operation,
                                                            Is_Reference);
            end if;
         else
            Error (ILLEGAL_OPERATOR_IN_CONCLUSION);
         end if;
      else
         Error (FAIL_CHUNCK_INACTIVE);
      end if;
   end Send_In_Conclusion;

   procedure Send_In_Excitatory_Condition (This : in Internal_Chunk) is
      use Errors_Manager;
      use Types_Index.Types_Index_Instance;
   begin
      if Is_Valid (This) then
         if (not (Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed)
             or else not ((Operators.Internal_Operators_With_Parameters (This.Operation).Information_Imposed
                           and This.Index = Operators.Internal_Operators_With_Parameters (This.Operation).Type_Value )
                          or Operators.Internal_Operators_With_Parameters (This.Operation).Information_Tolerance))
           and This.Operation > Last_Operator_Without_Parameter
           and not Operators.Internal_Operators_With_Parameters (This.Operation).Is_Conclusion then
            if Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed then
               Rule_Buffer.Writing.Set_Internal_Condition (Operators.Internal_Operators_With_Parameters (This.Operation).Type_Value,
                                                           This.Data,
                                                           This.Arrival_Time,
                                                           Premise_Properties.Excitatory_Evidence,
                                                           This.Operation);
            else
               Rule_Buffer.Writing.Set_Internal_Condition (This.Index,
                                                           This.Data,
                                                           This.Arrival_Time,
                                                           Premise_Properties.Excitatory_Evidence,
                                                           This.Operation);
            end if;
         else
            Error (ILLEGAL_OPERATOR_IN_CONDITION);
         end if;
      else
         Error (FAIL_CHUNCK_INACTIVE);
      end if;
   end Send_In_Excitatory_Condition;

   procedure Send_In_Inhibitory_Condition (This : in Internal_Chunk)is
      use Errors_Manager;
      use Types_Index.Types_Index_Instance;
   begin
      if Is_Valid (This) then
         if (not (Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed)
             or else not ((Operators.Internal_Operators_With_Parameters (This.Operation).Information_Imposed
                           and This.Index = Operators.Internal_Operators_With_Parameters (This.Operation).Type_Value )
                          or Operators.Internal_Operators_With_Parameters (This.Operation).Information_Tolerance))
           and This.Operation > Last_Operator_Without_Parameter
           and not Operators.Internal_Operators_With_Parameters (This.Operation).Is_Conclusion then
            if Operators.Internal_Operators_With_Parameters (This.Operation).Type_Imposed then
               Rule_Buffer.Writing.Set_Internal_Condition (Operators.Internal_Operators_With_Parameters (This.Operation).Type_Value,
                                                           This.Data,
                                                           This.Arrival_Time,
                                                           Premise_Properties.Inhibitory_Evidence,
                                                           This.Operation);
            else
               Rule_Buffer.Writing.Set_Internal_Condition (This.Index,
                                                           This.Data,
                                                           This.Arrival_Time,
                                                           Premise_Properties.Inhibitory_Evidence,
                                                           This.Operation);
            end if;
         else
            Error (ILLEGAL_OPERATOR_IN_CONDITION);
         end if;
      else
         Error (FAIL_CHUNCK_INACTIVE);
      end if;
   end Send_In_Inhibitory_Condition;

   procedure Set_Event_Position (This     : out Internal_Chunk;
                                 Position : in Internal_Event_Index) is
   begin
      This.Event_Position := Position;
   end Set_Event_Position;

   procedure Set_Operation (This      : out Internal_Chunk;
                            Operation : in Positive) is
   begin
      This.Operation := Operation;
   end Set_Operation;

   procedure Set_Temporal_Reference (This                    : in Internal_Chunk;
                                     Is_Reference_And_Origin : in Boolean) is
   begin
      if Is_Reference_And_Origin then
         Rule_Buffer.Writing.Set_Temporal_Reference (This.Arrival_Time);
      else
         Rule_Buffer.Writing.Set_Temporal_Reference (This.Capture_Time);
      end if;
   end Set_Temporal_Reference;

end Nomo.Interpreter.Plant.Internal_Chunks;
