--  Copyright (2008-2013) Cdric Coussinet (cedric.coussinet@nomoseed.net)
--
--  This program is free software: you can redistribute it and/or modify
--  it under the terms of the GNU Affero General Public License as published
--  by the Free Software Foundation, either version 3 of the License, or
--  (at your option) any later version.
--
--  This program is distributed in the hope that it will be useful,
--  but WITHOUT ANY WARRANTY; without even the implied warranty of
--  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
--  GNU Affero General Public License for more details.

--  You should have received a copy of the GNU Affero General Public License
--  along with this program. If not, see <http://www.gnu.org/licenses/>

with Nomo.Numerics.Accurately.Constants;

with Nomo.Numerics.Reals.Elementary_Functions;

package body Nomo.Internal_Messages.Premises.Loading is

   procedure Calculate_Specificity_Log (This : in out Internal_Premise);
   pragma Inline(Calculate_Specificity_Log);

   procedure Calculate_Specificity_Log (This : in out Internal_Premise) is
      use Numerics.Reals.Elementary_Functions;
      use Numerics.Accurately.Constants;
   begin
      case This.Tolerance is
         when IN_TN_CN =>
            This.Specificity_Log := 3.0 * Zero_Plus_Log;
         when IN_TN_CP =>
            This.Specificity_Log := Log_10 (This.Credibility_Tolerance / 2.0) + 2.0 * Zero_Plus_Log;
         when IN_TN_CI | IN_TI_CN | II_TN_CN=>
            This.Specificity_Log := Zero_Plus_Log;
         when IN_TP_CN =>
            This.Specificity_Log := Log_10 (This.Time_Span_Tolerance / 2.0) + 2.0 * Zero_Plus_Log;
         when IN_TP_CP =>
            This.Specificity_Log := Log_10 (This.Time_Span_Tolerance / 2.0) + Log_10 (This.Credibility_Tolerance / 2.0) + Zero_Plus_Log;
         when IN_TP_CI | II_TP_CN=>
            This.Specificity_Log := Log_10 (This.Time_Span_Tolerance / 2.0);
         when IN_TI_CP | II_TN_CP =>
            This.Specificity_Log := Log_10 (This.Credibility_Tolerance / 2.0);
         when IN_TI_CI | II_TI_CN | II_TN_CI=>
            This.Specificity_Log := Positive_Infinity_Log;
         when II_TI_CP =>
            This.Specificity_Log := Log_10 (This.Credibility_Tolerance / 2.0) + 2.0 * Zero_Plus_Log;
         when II_TP_CP =>
            This.Specificity_Log := Log_10 (This.Time_Span_Tolerance / 2.0) + Log_10 (This.Credibility_Tolerance / 2.0) + Positive_Infinity_Log;
         when II_TP_CI =>
            This.Specificity_Log := Log_10 (This.Time_Span_Tolerance / 2.0) + 2.0 * Positive_Infinity_Log;
         when II_TI_CI =>
            This.Specificity_Log := 3.0 * Positive_Infinity_Log;
      end case;
   end Calculate_Specificity_Log;

   procedure Set (This                  : out Internal_Premise;
                  Information           : in Positive_Integer;
                  Information_Tolerance : in Boolean;
                  Time_Span             : in Time_Interval;
                  Time_Span_Tolerance   : in Strictly_Positive_Real;
                  Credibility           : in Real_0_To_1;
                  Credibility_Tolerance : in Strictly_Positive_Real;
                  Fitting_Nbr           : in Count) is
   begin
      This.Information := Information;
      This.Time_Span := Time_Span;
      This.Credibility := Credibility;
      if Information_Tolerance then
         if Time_Span_Tolerance = Zero_Plus then
            if Credibility_Tolerance = Zero_Plus then
               This.Tolerance := II_TN_CN;
            elsif Credibility_Tolerance = Positive_Infinity then
               This.Tolerance := II_TN_CI;
            else
               This.Tolerance := II_TN_CP;
            end if;
         elsif Time_Span_Tolerance = Positive_Infinity then
            if Credibility_Tolerance = Zero_Plus then
               This.Tolerance := II_TI_CN;
            elsif Credibility_Tolerance = Positive_Infinity then
               This.Tolerance := II_TI_CI;
            else
               This.Tolerance := II_TI_CP;
            end if;
         else
            if Credibility_Tolerance = Zero_Plus then
               This.Tolerance := II_TP_CN;
            elsif Credibility_Tolerance = Positive_Infinity then
               This.Tolerance := II_TP_CI;
            else
               This.Tolerance := II_TP_CP;
            end if;
         end if;
      else
         if Time_Span_Tolerance = Zero_Plus then
            if Credibility_Tolerance = Zero_Plus then
               This.Tolerance := IN_TN_CN;
            elsif Credibility_Tolerance = Positive_Infinity then
               This.Tolerance := IN_TN_CI;
            else
               This.Tolerance := IN_TN_CP;
            end if;
         elsif Time_Span_Tolerance = Positive_Infinity then
            if Credibility_Tolerance = Zero_Plus then
               This.Tolerance := IN_TI_CN;
            elsif Credibility_Tolerance = Positive_Infinity then
               This.Tolerance := IN_TI_CI;
            else
               This.Tolerance := IN_TI_CP;
            end if;
         else
            if Credibility_Tolerance = Zero_Plus then
               This.Tolerance := IN_TP_CN;
            elsif Credibility_Tolerance = Positive_Infinity then
               This.Tolerance := IN_TP_CI;
            else
               This.Tolerance := IN_TP_CP;
            end if;
         end if;
      end if;
      This.Time_Span_Tolerance := Time_Span_Tolerance;
      if Time_Span_Tolerance /= Positive_Infinity and Time_Span_Tolerance /= Zero_Plus then
         This.Time_Span_Trend.Values_Sum := Time (Time_Span) * Time (Fitting_Nbr);
         This.Time_Span_Trend.Squares_Sum := (Time (Time_Span) ** 2 + Time (Time_Span_Tolerance))* Time (Fitting_Nbr);
      end if;
      This.Credibility_Tolerance := Credibility_Tolerance;
      if Credibility_Tolerance /= Positive_Infinity and Credibility_Tolerance /= Zero_Plus then
         This.Credibility_Trend.Values_Sum := Long_Real (Credibility * Real (Fitting_Nbr));
         This.Credibility_Trend.Squares_Sum := Long_Real (Credibility ** 2 + Credibility_Tolerance * Real (Fitting_Nbr));
      end if;
      Calculate_Specificity_Log(This);
   end Set;

end Nomo.Internal_Messages.Premises.Loading;
