<?xml version="1.0" encoding="utf-8" ?>
<!--
  Copyright (2010-2013) Cédric Coussinet (cedric.coussinet@nomoseed.net)

  This program is free software: you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published
  by the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program. If not, see <http://www.gnu.org/licenses/>
-->

<!--
Transforme un modèle suivant une grammaire basique ambigue en un modèle respectant une grammaire non ambigu.
Résoud partielement l'expression de "timespan" en remplacant "MAX" par la valeur de "time_span_limit".
-->
<xsl:stylesheet version="1.0" xmlns:xsl="http://www.w3.org/1999/XSL/Transform" xmlns:sdk="http://www.nomoseed.org/sdk" xmlns:program="http://www.nomoseed.org/program" xmlns:model="http://www.nomoseed.org/model" xmlns:project="http://www.nomoseed.org/project">
	<xsl:output method="xml" indent="yes"/>
  <xsl:variable name="timestep">
    <xsl:choose>
      <xsl:when test="/*/*/*[not(@value='auto') and local-name()='frequency']">
        <xsl:value-of select="floor(1000 div (/*/*/*[local-name()='frequency']/@value))"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="1"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
	<xsl:variable name="maximum_of_timespan">
    <xsl:variable name="maximum_of_timestep">
      <xsl:for-each select="//program:premise">
        <xsl:sort select="./program:timespan[not(contains(@value, 'M') or contains(@value, 'ms'))]/@value" order="descending"/>
        <xsl:if test="position()=1">
          <xsl:choose>
            <xsl:when test="./program:timespan/@value &gt; 0">
              <xsl:value-of select="number(./program:timespan/@value) * $timestep"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="0"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:if>
      </xsl:for-each>
    </xsl:variable>
    <xsl:variable name="maximum_of_ms">
      <xsl:for-each select="//program:premise">
        <xsl:sort select="./program:timespan[not(contains(@value, 'M')) and contains(@value, 'ms')]/@value" order="descending"/>
        <xsl:if test="position()=1">
          <xsl:choose>
            <xsl:when test="number(substring-before(./program:timespan/@value, 'ms')) &gt; 0">
              <xsl:value-of select="number(substring-before(./program:timespan/@value, 'ms'))"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="0"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:if>
      </xsl:for-each>
    </xsl:variable>
    <xsl:choose>
      <xsl:when test="$maximum_of_ms &gt; $maximum_of_timestep">
        <xsl:value-of select="$maximum_of_ms"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="$maximum_of_timestep"/>
      </xsl:otherwise>
    </xsl:choose>
	</xsl:variable>
  <xsl:variable name="time_span_limit">
    <xsl:choose>
      <xsl:when test="/*/*/*[not(@value='auto') and local-name()='time_span_limit']">
          <xsl:choose>
            <xsl:when test="contains(/*/*/*[local-name()='time_span_limit']/@value, 'ms')">
              <xsl:value-of select="number(substring-before(/*/*/*[local-name()='time_span_limit']/@value,'ms'))"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="format-number(number(/*/*/*[local-name()='time_span_limit']/@value) * $timestep, '#')"/>
            </xsl:otherwise>
          </xsl:choose>
      </xsl:when>
      <xsl:when test="/*/*/*[@value='auto' and local-name()='time_span_limit']">
        <xsl:value-of select="$maximum_of_timespan"/>
      </xsl:when>
      <xsl:otherwise>
        <xsl:value-of select="format-number(4611686018427387904,'#')"/>
      </xsl:otherwise>
    </xsl:choose>
  </xsl:variable>
  <xsl:variable name="time_span_max">
     <xsl:value-of select="format-number(floor($time_span_limit div $timestep) * $timestep,'#')"/>
  </xsl:variable>
	<xsl:template match="model:model">
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
      <xsl:if test="@xml:base">
        <xsl:attribute name="xml:base">
          <xsl:value-of select="@xml:base"/>
        </xsl:attribute>
      </xsl:if>
			<xsl:copy-of select="./project:header"/>
			<xsl:choose>
				<xsl:when test="./model:base">
					<xsl:element name="base" namespace="http://www.nomoseed.org/model">
						<xsl:apply-templates select="./model:base/model:model"/>
					</xsl:element>
					<xsl:apply-templates select="./model:definition">
						<xsl:with-param name="model" select="@name"/>
						<xsl:with-param name="base" select="./model:base/model:model/@name"/>
					</xsl:apply-templates>
				</xsl:when>
				<xsl:otherwise>
					<xsl:element name="base" namespace="http://www.nomoseed.org/model">
						<xsl:element name="null" namespace="http://www.nomoseed.org/model">
							<xsl:attribute name="name">
								<xsl:value-of select="'*'"/>
							</xsl:attribute>
						</xsl:element>
					</xsl:element>
					<xsl:apply-templates select="./model:definition">
						<xsl:with-param name="model" select="@name"/>
						<xsl:with-param name="base" select="'*'"/>
					</xsl:apply-templates>
				</xsl:otherwise>
			</xsl:choose>
			<xsl:copy-of select="./sdk:sdk"/>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:definition">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:apply-templates select="./model:*">
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:perceptive_structure">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declarations" namespace="http://www.nomoseed.org/model">
				<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
					<xsl:attribute name="category">
						<xsl:value-of select="'perception'"/>
					</xsl:attribute>
					<xsl:attribute name="type">
						<xsl:value-of select="@name"/>
					</xsl:attribute>
					<xsl:attribute name="model">
						<xsl:value-of select="$model"/>
					</xsl:attribute>
				</xsl:element>
				<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
					<xsl:attribute name="category">
						<xsl:value-of select="'input'"/>
					</xsl:attribute>
					<xsl:attribute name="type">
						<xsl:value-of select="@name"/>
					</xsl:attribute>
					<xsl:attribute name="model">
						<xsl:value-of select="$model"/>
					</xsl:attribute>
				</xsl:element>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by | ./model:predicted_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:command_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
				<xsl:with-param name="components" select="./model:components/model:component"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
			<xsl:element name="inputs" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:components/project:annotation"/>
				<xsl:apply-templates select="./model:components/model:component" mode="index">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
				<xsl:apply-templates select="./model:components/model:component[1]" mode="component">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:item">
		<xsl:param name="type"/>
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:element name="item" namespace="http://www.nomoseed.org/model">
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:attribute name="type">
				<xsl:value-of select="$type"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:component" mode="index">
		<xsl:param name="type"/>
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:element name="index" namespace="http://www.nomoseed.org/model">
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:attribute name="type">
				<xsl:value-of select="$type"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:variable name="pos">
				<xsl:number count="model:component" format="1" level="single"/>
			</xsl:variable>
			<xsl:attribute name="pos">
				<xsl:value-of select="$pos"/>
			</xsl:attribute>
			<xsl:attribute name="next">
				<xsl:choose>
					<xsl:when test="following-sibling::model:component">
						<xsl:value-of select="$pos + 1"/>
					</xsl:when>
					<xsl:otherwise>
						<xsl:value-of select="'*'"/>
					</xsl:otherwise>
				</xsl:choose>
			</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:component" mode="component">
		<xsl:param name="type"/>
		<xsl:param name="model"/>
		<xsl:element name="component" namespace="http://www.nomoseed.org/model">
			<xsl:attribute name="type">
				<xsl:value-of select="$type"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:variable name="pos">
				<xsl:number count="model:component" format="1" level="single"/>
			</xsl:variable>
			<xsl:attribute name="pos">
				<xsl:value-of select="$pos"/>
			</xsl:attribute>
			<xsl:choose>
				<xsl:when test="following-sibling::model:component">
					<xsl:attribute name="next">
						<xsl:value-of select="$pos + 1"/>
					</xsl:attribute>
					<xsl:apply-templates select="following-sibling::model:component[1]" mode="component">
						<xsl:with-param name="type" select="$type"/>
						<xsl:with-param name="model" select="$model"/>
					</xsl:apply-templates>
				</xsl:when>
				<xsl:otherwise>
					<xsl:attribute name="next">
						<xsl:value-of select="'*'"/>
					</xsl:attribute>
					<xsl:element name="end" namespace="http://www.nomoseed.org/model">
						<xsl:attribute name="pos">
							<xsl:value-of select="'*'"/>
						</xsl:attribute>
					</xsl:element>
				</xsl:otherwise>
			</xsl:choose>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:rewarded_by | model:predicted_by">
		<xsl:param name="in"/>
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:attribute name="in">
				<xsl:value-of select="$in"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:conception_type" mode="in">
		<xsl:param name="in"/>
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:attribute name="in">
				<xsl:value-of select="$in"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
				<xsl:attribute name="category">
					<xsl:value-of select="'conception'"/>
				</xsl:attribute>
				<xsl:attribute name="type">
					<xsl:value-of select="@name"/>
				</xsl:attribute>
				<xsl:attribute name="model">
					<xsl:value-of select="$model"/>
				</xsl:attribute>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by | ./model:predicted_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:command_type" mode="in">
		<xsl:param name="in"/>
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:param name="components"/>
		<xsl:copy>
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:attribute name="in">
				<xsl:value-of select="$in"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:attribute name="name">
				<xsl:value-of select="$in"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
				<xsl:attribute name="category">
					<xsl:value-of select="'command'"/>
				</xsl:attribute>
				<xsl:attribute name="type">
					<xsl:value-of select="$in"/>
				</xsl:attribute>
				<xsl:attribute name="model">
					<xsl:value-of select="$model"/>
				</xsl:attribute>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by">
				<xsl:with-param name="in" select="$in"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="$in"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="$in"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
			<xsl:element name="outputs" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="../model:components/project:annotation"/>
				<xsl:apply-templates select="$components" mode="index">
					<xsl:with-param name="type" select="$in"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
				<xsl:apply-templates select="$components[1]" mode="component">
					<xsl:with-param name="type" select="$in"/>
					<xsl:with-param name="model" select="$model"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:command_type">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
				<xsl:attribute name="category">
					<xsl:value-of select="'command'"/>
				</xsl:attribute>
				<xsl:attribute name="type">
					<xsl:value-of select="@name"/>
				</xsl:attribute>
				<xsl:attribute name="model">
					<xsl:value-of select="$model"/>
				</xsl:attribute>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
			<xsl:element name="outputs" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:components/project:annotation"/>
				<xsl:apply-templates select="./model:components/model:component" mode="index">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
				<xsl:apply-templates select="./model:components/model:component[1]" mode="component">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:conception_type">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
				<xsl:attribute name="category">
					<xsl:value-of select="'conception'"/>
				</xsl:attribute>
				<xsl:attribute name="type">
					<xsl:value-of select="@name"/>
				</xsl:attribute>
				<xsl:attribute name="model">
					<xsl:value-of select="$model"/>
				</xsl:attribute>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by | ./model:predicted_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:predictive_structure">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:attribute name="target">
				<xsl:apply-templates select="parent::model:definition" mode="target">
					<xsl:with-param name="name" select="@name"/>
				</xsl:apply-templates>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declarations" namespace="http://www.nomoseed.org/model">
				<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
					<xsl:attribute name="category">
						<xsl:value-of select="'prediction'"/>
					</xsl:attribute>
					<xsl:attribute name="type">
						<xsl:value-of select="@name"/>
					</xsl:attribute>
					<xsl:attribute name="model">
						<xsl:value-of select="$model"/>
					</xsl:attribute>
				</xsl:element>
				<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
					<xsl:attribute name="category">
						<xsl:value-of select="'landmark'"/>
					</xsl:attribute>
					<xsl:attribute name="type">
						<xsl:value-of select="@name"/>
					</xsl:attribute>
					<xsl:attribute name="model">
						<xsl:value-of select="$model"/>
					</xsl:attribute>
				</xsl:element>
				<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
					<xsl:attribute name="category">
						<xsl:value-of select="'check'"/>
					</xsl:attribute>
					<xsl:attribute name="type">
						<xsl:value-of select="@name"/>
					</xsl:attribute>
					<xsl:attribute name="model">
						<xsl:value-of select="$model"/>
					</xsl:attribute>
				</xsl:element>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:definition" mode="target">
		<xsl:param name="name"/>
		<xsl:value-of select="//model:*[./model:predicted_by/@name=$name]/@name"/>
	</xsl:template>
	<xsl:template match="model:reward_type">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
				<xsl:attribute name="category">
					<xsl:value-of select="'reward'"/>
				</xsl:attribute>
				<xsl:attribute name="type">
					<xsl:value-of select="@name"/>
				</xsl:attribute>
				<xsl:attribute name="model">
					<xsl:value-of select="$model"/>
				</xsl:attribute>
			</xsl:element>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:landmark_type">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:copy-of select="./project:annotation"/>
			<xsl:element name="declaration" namespace="http://www.nomoseed.org/model">
				<xsl:attribute name="category">
					<xsl:value-of select="'landmark'"/>
				</xsl:attribute>
				<xsl:attribute name="type">
					<xsl:value-of select="@name"/>
				</xsl:attribute>
				<xsl:attribute name="model">
					<xsl:value-of select="$model"/>
				</xsl:attribute>
			</xsl:element>
			<xsl:apply-templates select="./model:rewarded_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:apply-templates select="./model:conception_type" mode="in">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="items" namespace="http://www.nomoseed.org/model">
				<xsl:copy-of select="./model:items/project:annotation"/>
				<xsl:apply-templates select="./model:items/model:item">
					<xsl:with-param name="type" select="@name"/>
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:plant_structure">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:copy>
			<xsl:attribute name="name">
				<xsl:value-of select="'plant'"/>
			</xsl:attribute>
			<xsl:attribute name="extension">
				<xsl:value-of select="'true'"/>
			</xsl:attribute>
			<xsl:apply-templates select="./model:rewarded_by">
				<xsl:with-param name="in" select="@name"/>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
			<xsl:element name="registers_structure" namespace="http://www.nomoseed.org/plant">
				<xsl:apply-templates select="./model:input">
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
				<xsl:apply-templates select="./model:other">
					<xsl:with-param name="model" select="$model"/>
					<xsl:with-param name="base" select="$base"/>
				</xsl:apply-templates>
			</xsl:element>
		</xsl:copy>
	</xsl:template>
	<xsl:template match="model:input | model:other | model:scope_type | model:operator_type">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:element name="{local-name(.)}" namespace="http://www.nomoseed.org/plant">
			<xsl:apply-templates>
				<xsl:with-param name="model" select="$model"/>
				<xsl:with-param name="base" select="$base"/>
			</xsl:apply-templates>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:scope_type/model:item | model:operator_type/model:item">
		<xsl:param name="model"/>
		<xsl:param name="base"/>
		<xsl:element name="item" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="base">
				<xsl:value-of select="$base"/>
			</xsl:attribute>
			<xsl:attribute name="name">
				<xsl:value-of select="@name"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="$model"/>
			</xsl:attribute>
			<xsl:apply-templates>
			</xsl:apply-templates>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:item/model:target">
		<xsl:element name="target" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="category">
				<xsl:value-of select="@category"/>
			</xsl:attribute>
			<xsl:attribute name="type">
				<xsl:value-of select="@type"/>
			</xsl:attribute>
			<xsl:attribute name="model">
				<xsl:value-of select="@model"/>
			</xsl:attribute>
			<xsl:attribute name="item">
				<xsl:value-of select="ancestor::model:item/@name"/>
			</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:other/model:operator_type/model:item/model:premise">
		<xsl:param name="model"/>
		<xsl:variable name="item" select="ancestor::model:item/@name"/>
		<xsl:element name="premise" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="information_tolerance">
				<xsl:value-of select="model:information/@tolerance"/>
			</xsl:attribute>
			<xsl:attribute name="credibility_tolerance">
				<xsl:value-of select="model:credibility/@tolerance"/>
			</xsl:attribute>
			<xsl:attribute name="timespan_tolerance">
        <xsl:choose>
          <xsl:when test="contains(model:timespan/@tolerance, 'period')">
            <xsl:value-of select="number(substring-before(model:timespan/@tolerance,'period')) * $timestep"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="model:timespan/@tolerance"/>
          </xsl:otherwise>
        </xsl:choose>
			</xsl:attribute>
			<xsl:attribute name="item">
				<xsl:value-of select="$item"/>
			</xsl:attribute>
      <xsl:call-template name="targetContent">
        <xsl:with-param name="item" select="$item"/>
        <xsl:with-param name="model" select="$model"/>
      </xsl:call-template>
			<xsl:apply-templates select="./model:credibility[@value]"/>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:premise/model:credibility[@value]">
		<xsl:element name="{local-name(.)}" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="value">
				<xsl:value-of select="@value"/>
			</xsl:attribute>
			<xsl:attribute name="item">
				<xsl:value-of select="ancestor::model:item/@name"/>
			</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:premise/model:timespan[@value]">
		<xsl:element name="timespan" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="value">
        <xsl:choose>
          <xsl:when test="contains(@value, 'MIN')">
            <xsl:choose>
              <xsl:when test="not(contains(@value, '+'))">
                <xsl:value-of select="'MIN'"/>
              </xsl:when>
              <xsl:when test="contains(@value, 'ms')">
                <xsl:value-of select="concat('MIN+', number(substring-before(substring-after(@value,'+'),'ms')))"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="concat('MIN+', format-number(number(concat('0',number(substring-after(@value,'+')))) * $timestep, '#'))"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:when>
          <xsl:when test="contains(@value, 'MAX')">
            <xsl:choose>
              <xsl:when test="contains(@value, 'ms')">
                <xsl:value-of select="format-number($time_span_max - number(substring-before(substring-after(@value,'-'),'ms')), '#')"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="format-number($time_span_max - number(concat('0',number(substring-after(@value,'-')))) * $timestep, '#')"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:when>
          <xsl:when test="contains(@value, 'ms')">
            <xsl:value-of select="substring-before(@value,'ms')"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="number(@value) * $timestep"/>
          </xsl:otherwise>
        </xsl:choose>
			</xsl:attribute>
			<xsl:attribute name="item">
				<xsl:value-of select="ancestor::model:item/@name"/>
			</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:information[@delay]">
		<xsl:element name="delay" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="value">
        <xsl:choose>
          <xsl:when test="contains(@value, 'MAX')">
            <xsl:choose>
              <xsl:when test="not(contains(@value, '-'))">
                <xsl:value-of select="'MAX'"/>
              </xsl:when>
              <xsl:when test="contains(@value, 'ms')">
                <xsl:value-of select="concat('MAX-', number(substring-before(substring-after(@value,'+'),'ms')))"/>
              </xsl:when>
              <xsl:otherwise>
                <xsl:value-of select="concat('MAX-', format-number(number(concat('0',number(substring-after(@value,'+')))) * $timestep, '#'))"/>
              </xsl:otherwise>
            </xsl:choose>
          </xsl:when>
          <xsl:when test="contains(@delay, 'ms')">
            <xsl:value-of select="number(substring-before(@delay,'ms'))"/>
          </xsl:when>
          <xsl:otherwise>
            <xsl:value-of select="format-number(number(@delay) * $timestep, '#')"/>
          </xsl:otherwise>
        </xsl:choose>
			</xsl:attribute>
			<xsl:attribute name="item">
				<xsl:value-of select="ancestor::model:item/@name"/>
			</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:input/model:operator_type/model:item/model:premise">
		<xsl:element name="premise" namespace="http://www.nomoseed.org/plant">
			<xsl:attribute name="information_tolerance">
				<xsl:value-of select="model:informations/@tolerance"/>
			</xsl:attribute>
		</xsl:element>
	</xsl:template>
	<xsl:template match="model:conclusion">
		<xsl:param name="model"/>
		<xsl:element name="conclusion" namespace="http://www.nomoseed.org/plant">
			<xsl:variable name="item" select="ancestor::model:item/@name"/>
			<xsl:attribute name="relevance">
				<xsl:value-of select="@relevance"/>
			</xsl:attribute>
			<xsl:attribute name="fitting_nbr">
				<xsl:value-of select="@fitting_nbr"/>
			</xsl:attribute>
			<xsl:attribute name="item">
				<xsl:value-of select="$item"/>
			</xsl:attribute>
      <xsl:call-template name="targetContent">
        <xsl:with-param name="item" select="$item"/>
        <xsl:with-param name="model" select="$model"/>
      </xsl:call-template>
		</xsl:element>
	</xsl:template>
  <xsl:template name="targetContent">
    <xsl:param name="item"/>
    <xsl:param name="model"/>
    <xsl:choose>
      <xsl:when test="./model:delay/@value &gt; 0 or ./model:timespan/@value &lt; 0 or ./model:information[@value]">
        <xsl:variable name="tag">
          <xsl:choose>
            <xsl:when test="@category = 'scope' or @category = 'operator'">
          <xsl:choose>
            <xsl:when test="@type = 'register_00'">
              <xsl:value-of select="concat('input_',@category)"/>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="concat('other_',@category)"/>
            </xsl:otherwise>
          </xsl:choose>
            </xsl:when>
            <xsl:otherwise>
              <xsl:value-of select="@category"/>
            </xsl:otherwise>
          </xsl:choose>
        </xsl:variable>
        <xsl:element name="{$tag}" namespace="http://www.nomoseed.org/plant">
          <xsl:if test="./model:information[@value]">
            <xsl:element name="information" namespace="http://www.nomoseed.org/plant">
              <xsl:attribute name="model">
                <xsl:choose>
                  <xsl:when test="contains (@model, 'this')">
                    <xsl:value-of select="$model"/>
                  </xsl:when>
                  <xsl:otherwise>
                    <xsl:value-of select="'plant'"/>
                  </xsl:otherwise>
                </xsl:choose>
              </xsl:attribute>
              <xsl:attribute name="value">
                <xsl:value-of select="./model:information/@value"/>
              </xsl:attribute>
              <xsl:attribute name="type">
                <xsl:value-of select="@type"/>
              </xsl:attribute>
              <xsl:attribute name="item">
                <xsl:value-of select="ancestor::model:item/@name"/>
              </xsl:attribute>
            </xsl:element>
          </xsl:if>
          <xsl:apply-templates select="./model:information[@delay] | ./model:timespan[@value]"/>
        </xsl:element>
      </xsl:when>
      <xsl:otherwise>
        <xsl:if test="@model or @type or @category">
          <xsl:element name="target" namespace="http://www.nomoseed.org/plant">
            <xsl:attribute name="category">
              <xsl:value-of select="@category"/>
            </xsl:attribute>
            <xsl:attribute name="type">
              <xsl:value-of select="@type"/>
            </xsl:attribute>
            <xsl:attribute name="model">
              <xsl:choose>
                <xsl:when test="contains (@model, 'this')">
                  <xsl:value-of select="$model"/>
                </xsl:when>
                <xsl:otherwise>
                  <xsl:value-of select="'plant'"/>
                </xsl:otherwise>
              </xsl:choose>
            </xsl:attribute>
            <xsl:attribute name="item">
              <xsl:value-of select="$item"/>
            </xsl:attribute>
          </xsl:element>
        </xsl:if>
        <xsl:apply-templates select="./model:information[@delay] | ./model:timespan[@value]"/>
      </xsl:otherwise>
    </xsl:choose>
	</xsl:template>
</xsl:stylesheet>
